<?php
/*--------------------------------------------------------------------
 AttributeModifierReader.php 2020-2-18
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------------*/
declare(strict_types=1);

namespace Gambio\Shop\Attributes\ProductModifiers\Database\Readers;

use CI_DB_query_builder;
use Gambio\Shop\Attributes\ProductModifiers\Database\ValueObjects\AttributeGroupIdentifier;
use Gambio\Shop\Attributes\ProductModifiers\Database\ValueObjects\AttributeModifierIdentifier;
use Gambio\Shop\Product\ValueObjects\ProductId;
use Gambio\Shop\ProductModifiers\Database\Core\DTO\Modifiers\ModifierDTOBuilderInterface;
use Gambio\Shop\ProductModifiers\Database\Core\DTO\Modifiers\ModifierDTOCollection;
use Gambio\Shop\ProductModifiers\Database\Core\DTO\Modifiers\ModifierDTOCollectionInterface;
use Gambio\Shop\ProductModifiers\Database\Core\Readers\Interfaces\ModifierReaderCompositeInterface;
use IdType;

/**
 * Class AttributeModifierReader
 * @package Gambio\Shop\Attributes\ProductModifiers\Database\Readers
 */
class AttributeModifierReader implements ModifierReaderCompositeInterface
{
    /**
     * @var ModifierDTOBuilderInterface
     */
    private $builder;
    /**
     * @var CI_DB_query_builder
     */
    private $queryBuilder;


    /**
     * AttributeModifierReader constructor.
     *
     * @param CI_DB_query_builder $queryBuilder
     * @param ModifierDTOBuilderInterface $builder
     */
    public function __construct(CI_DB_query_builder $queryBuilder, ModifierDTOBuilderInterface $builder)
    {
        $this->queryBuilder = $queryBuilder;
        $this->builder = $builder;
    }


    /**
     * @inheritDoc
     */
    public function getModifierByProduct(ProductId $id, IdType $languageId): ModifierDTOCollectionInterface
    {
        $result = new ModifierDTOCollection();

        $sql = "SELECT
                    po.products_options_id,
                    pov.products_options_values_id,
                    pov.products_options_values_name,
                    po.products_option_display_type,    
                    pa.options_values_price,
                    pa.price_prefix,    
                    (select max(gm_filename) from products_options_values where products_options_values.`products_options_values_id` = pov.`products_options_values_id`) gm_filename

                    FROM products_attributes pa
                        INNER JOIN products_options po 
                                ON po.products_options_id = pa.options_id 
                        INNER JOIN products_options_values_to_products_options povtpo 
                                ON po.products_options_id = povtpo.products_options_id
                               AND povtpo.products_options_values_id = pa.options_values_id
                        INNER JOIN products_options_values pov 
                                ON povtpo.products_options_values_id = pov.products_options_values_id
                               AND pov.language_id = po.language_id
                    WHERE pa.products_id = {$id->value()} 
                      AND po.language_id = {$languageId->asInt()}                      
                    GROUP BY po.products_options_id, pov.products_options_values_id, pov.products_options_values_name, po.products_option_display_type, pa.sortorder
                    ORDER BY pa.sortorder";

        $query = $this->queryBuilder->query($sql)->result_array();

        foreach ($query as $item) {
            $result->addModifier(
                $this->builder
                    ->withId(new AttributeModifierIdentifier($item['products_options_values_id']))
                    ->withGroupId(new AttributeGroupIdentifier((int)$item['products_options_id']))
                    ->withType($item['products_option_display_type'])
                    ->withName($item['products_options_values_name'])
                    ->withPricePrefix($item['price_prefix'])
                    ->withPrice((float)$item['options_values_price'])
                    ->withSource('attribute')
                    ->withImage($item['gm_filename'] ? 'product_images/attribute_images/' . $item['gm_filename'] : '')
                    ->build()
            );
        }
        return $result;
    }

    /**
     * @param $item
     * @return string
     */
    protected function parsePrice($item): string
    {
        $price = (float)$item['options_values_price'];
        if ($price) {
            return $item['price_prefix'] . ' ' . $price;
        }

        return '';
    }
}